#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
"""
Unit tests for the schooltool.intervention.adapters.
"""
import unittest
import doctest

from zope.app.testing.placelesssetup import setUp, tearDown
from zope.component import provideAdapter
from zope.interface import implements

from schooltool.app.interfaces import ISchoolToolApplication
from schooltool.intervention import adapters


def doctest_PersonRemovedSubsciber():
    """
    First we need some stubs.

        >>> class PersonStub:
        ...     def __init__(self, username):
        ...         self.username = username

        >>> class AppStub(dict):
        ...    implements(ISchoolToolApplication)

    We need to create an application, intervention root, intervention school
    year and a couple intervention students.

        >>> app = AppStub()
        >>> app['schooltool.interventions'] = root = {}
        >>> root['2009'] = year = {}
        >>> year['student1'] = 'Student One'
        >>> year['student2'] = 'Student Two'

    We need an adapter to get the app.

        >>> provideAdapter(lambda x: app, (None,), ISchoolToolApplication)

    We'll load the subscriber with student1 and see how the intervention
    student object will disappear.

        >>> student1 = PersonStub('student1')
        >>> subscriber = adapters.PersonRemovedSubsciber(None, student1)
        >>> subscriber()
        >>> sorted(year.keys())
        ['student2']
    """


def doctest_SchoolYearRemovedSubsciber():
    """
    First we need some stubs.

        >>> class SchoolYearStub:
        ...     def __init__(self, name):
        ...         self.__name__ = name

        >>> class AppStub(dict):
        ...    implements(ISchoolToolApplication)

    We need to create an application, an intervention root and a couple
    intervention schoolyears.

        >>> app = AppStub()
        >>> app['schooltool.interventions'] = root = {}
        >>> root['2008'] = {}
        >>> root['2009'] = {}

    We need an adapter to get the app.

        >>> provideAdapter(lambda x: app, (None,), ISchoolToolApplication)

    We'll load the subscriber with student1 and see how the intervention
    student object will disappear.

        >>> year = SchoolYearStub('2009')
        >>> subscriber = adapters.SchoolYearRemovedSubsciber(None, year)
        >>> subscriber()
        >>> sorted(root.keys())
        ['2008']
    """


def test_suite():
    optionflags = doctest.NORMALIZE_WHITESPACE | doctest.ELLIPSIS
    suite = doctest.DocTestSuite(optionflags=optionflags,
                                 setUp=setUp, tearDown=tearDown)
    return unittest.TestSuite([suite])


if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')

