/****************************************************************************
**
** Modifications Copyright (C) 2016 Kevin B. Hendricks, Stratford, ON Canada
**
** Copyright (C) 2013 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the examples of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:BSD$
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Digia Plc and its Subsidiary(-ies) nor the names
**     of its contributors may be used to endorse or promote products derived
**     from this software without specific prior written permission.
**
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QtWidgets>

#include "Dialogs/TreeItem.h"
#include "Dialogs/TreeModel.h"

#if QT_VERSION >= QT_VERSION_CHECK(5, 15, 0)
    #define QT_ENUM_SKIPEMPTYPARTS Qt::SkipEmptyParts
    #define QT_ENUM_KEEPEMPTYPARTS Qt::KeepEmptyParts
#else
    #define QT_ENUM_SKIPEMPTYPARTS QString::SkipEmptyParts
    #define QT_ENUM_KEEPEMPTYPARTS QString::KeepEmptyParts
#endif

const QString _GS = QString(QChar(29));
const QString _RS = QString(QChar(30));
const QString _US = QString(QChar(31));
const QString _IN = QString("  ");


TreeModel::TreeModel(const QStringList &headers, const QString &data, QObject *parent)
    : QAbstractItemModel(parent)
{
    QVector<QVariant> rootData;
    foreach (QString header, headers)
        rootData << header;

    rootItem = new TreeItem(rootData);
    QStringList datalist = data.split(_RS, QT_ENUM_SKIPEMPTYPARTS);
    setupModelData(datalist, rootItem);
}


TreeModel::~TreeModel()
{
    delete rootItem;
}


int TreeModel::columnCount(const QModelIndex & /* parent */) const
{
    return rootItem->columnCount();
}

QVariant TreeModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();

    if (role != Qt::DisplayRole && role != Qt::EditRole && role != Qt::ToolTipRole)
        return QVariant();

    TreeItem *item = getItem(index);
    if (role == Qt::ToolTipRole) {
        return item->tip(index.column());
    }
    return item->data(index.column());
}

// Note only column 1 values are user editable, column 0 is not
Qt::ItemFlags TreeModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
      return Qt::ItemFlags();

    if (index.column() == 0) {
        return QAbstractItemModel::flags(index);
    }
    return Qt::ItemIsEditable | QAbstractItemModel::flags(index);
}

TreeItem *TreeModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        TreeItem *item = static_cast<TreeItem*>(index.internalPointer());
        if (item)
            return item;
    }
    return rootItem;
}


QVariant TreeModel::headerData(int section, Qt::Orientation orientation,
                               int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole)
        return rootItem->data(section);

    return QVariant();
}

QModelIndex TreeModel::index(int row, int column, const QModelIndex &parent) const
{
    if (parent.isValid() && parent.column() != 0)
        return QModelIndex();

    TreeItem *parentItem = getItem(parent);

    TreeItem *childItem = parentItem->child(row);
    if (childItem)
        return createIndex(row, column, childItem);
    else
        return QModelIndex();
}


bool TreeModel::insertRows(int position, int rows, const QModelIndex &parent)
{
    TreeItem *parentItem = getItem(parent);
    bool success;

    beginInsertRows(parent, position, position + rows - 1);
    success = parentItem->insertChildren(position, rows, rootItem->columnCount());
    endInsertRows();

    return success;
}



QModelIndex TreeModel::parent(const QModelIndex &index) const
{
    if (!index.isValid())
        return QModelIndex();

    TreeItem *childItem = getItem(index);
    TreeItem *parentItem = childItem->parent();

    if (parentItem == rootItem)
        return QModelIndex();

    return createIndex(parentItem->childNumber(), 0, parentItem);
}


bool TreeModel::removeRows(int position, int rows, const QModelIndex &parent)
{
    TreeItem *parentItem = getItem(parent);
    bool success = true;

    beginRemoveRows(parent, position, position + rows - 1);
    success = parentItem->removeChildren(position, rows);
    endRemoveRows();

    return success;
}


bool TreeModel::moveRowUp(int position, const QModelIndex &parent)
{
    if (position == 0) return false;
  
    TreeItem *parentItem = getItem(parent);
    bool success = true;
    if (!beginMoveRows(parent, position, position, parent, position-1)) {
        return false;
    }
    success = parentItem->moveChildUp(position);
    endMoveRows();

    return success;
}


bool TreeModel::moveRowDown(int position, const QModelIndex &parent)
{
    TreeItem *parentItem = getItem(parent);
    if (position == parentItem->childCount()-1) return false;
    bool success = true;
    if (!beginMoveRows(parent, position, position, parent, position+2)) {
        return false;
    }
    success = parentItem->moveChildDown(position);
    endMoveRows();

    return success;
}


int TreeModel::rowCount(const QModelIndex &parent) const
{
    TreeItem *parentItem = getItem(parent);
    return parentItem->childCount();
}

bool TreeModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    if ((role != Qt::EditRole) && (role !=Qt::ToolTipRole))
        return false;

    TreeItem *item = getItem(index);

    bool result;
    if (role == Qt::ToolTipRole) {
        result = item->setTips(index.column(), value);
    } else {    
        result = item->setData(index.column(), value);
    }
    if (result)
        emit dataChanged(index, index);

    return result;
}

bool TreeModel::setHeaderData(int section, Qt::Orientation orientation,
                              const QVariant &value, int role)
{
    if (role != Qt::EditRole || orientation != Qt::Horizontal)
        return false;

    bool result = rootItem->setData(section, value);

    if (result)
        emit headerDataChanged(orientation, section, section);

    return result;
}

void TreeModel::setupModelData(const QStringList &lines, TreeItem *parent)
{
    QList<TreeItem*> parents;
    QList<int> indentations;
    parents << parent;
    indentations << 0;

    int number = 0;

    while (number < lines.count()) {
        int position = 0;
        while (position < lines[number].length()) {
            if (lines[number].mid(position, 1) != " ")
                break;
            ++position;
        }

        QString lineData = lines[number].mid(position).trimmed();

        if (!lineData.isEmpty()) {
            // Read the column data from the rest of the line.
            QStringList columnStrings = lineData.split(_US, QT_ENUM_SKIPEMPTYPARTS);
            QVector<QVariant> columnData;
            QVector<QVariant> tipData;
            for (int column = 0; column < columnStrings.count(); ++column) {
                QStringList parts = columnStrings[column].split(_GS, QT_ENUM_KEEPEMPTYPARTS);
                columnData << parts.at(0);
                tipData << parts.at(1);
            }
            if (position > indentations.last()) {
                // The last child of the current parent is now the new parent
                // unless the current parent has no children.

                if (parents.last()->childCount() > 0) {
                    parents << parents.last()->child(parents.last()->childCount()-1);
                    indentations << position;
                }
            } else {
                while (position < indentations.last() && parents.count() > 0) {
                    parents.pop_back();
                    indentations.pop_back();
                }
            }

            // Append a new item to the current parent's list of children.
            TreeItem *parent = parents.last();
            parent->insertChildren(parent->childCount(), 1, rootItem->columnCount());
            for (int column = 0; column < columnData.size(); ++column) {
                parent->child(parent->childCount() - 1)->setData(column, columnData[column]);
                parent->child(parent->childCount() - 1)->setTips(column, tipData[column]);
            }
        }

        ++number;
    }
}


QStringList TreeModel::getModelData(TreeItem * node, int lvl) 
{
    QStringList reslst;
    int cols = rootItem->columnCount();
    QString indent = _IN.repeated(lvl);
    QString res = indent + (node->data(0)).toString();
    for (int col=1; col < cols; col++) {
      res = res + _US + (node->data(col)).toString();
    }
    reslst.append(res);
    int numchild = node->childCount();
    for (int row=0; row < numchild; row++) {
      TreeItem* childItem=node->child(row);
      reslst.append(getModelData(childItem, lvl+1));
    }
    return reslst;
}


QString TreeModel::getAllModelData() 
{
    QStringList results;
    int numchild = rootItem->childCount();
    for (int row=0; row < numchild; row++) {
        TreeItem* childItem=rootItem->child(row);
        results.append(getModelData(childItem, 0));
    }
    return results.join(_RS);
}
