/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include <vdb/extern.h>

#define TRACK_REFERENCES 0
/* should match dbmgr.c and/or wdbmgr.c */

#define KONST const
#include "dbmgr-priv.h"
#undef KONST

#include "schema-priv.h"
#include "linker-priv.h"

#include <vdb/manager.h>
#include <vdb/database.h>
#include <vdb/vdb-priv.h>
#include <kdb/manager.h>
#include <kfg/config.h>
#include <kfs/directory.h>
#include <kfs/dyload.h>
#include <klib/text.h>
#include <klib/rc.h>
#include <sysalloc.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#if 1
#define DEBUG_PRINT(fmt, ...) { do { } while(0); }
#else
#define DEBUG_PRINT(fmt, ...) fprintf(stderr, "%s - " fmt "\n", __func__, __VA_ARGS__)
#endif

/*--------------------------------------------------------------------------
 * VDBMem
 */

/* Whack
 */
static
void CC VDBMemWhack ( DLNode *n, void *ignore )
{
    DEBUG_PRINT("freeing mem", 0);
    free ( n );
}


/* Release
 */
void CC VDBMemRelease ( DLNode *item, void *data )
{
    VDBMem *self = ( VDBMem* ) item;
    VDBManager *mgr = data;

    if ( self != NULL )
    {
        if ( mgr == NULL || mgr -> pcount >= mgr -> plimit ) {
            if (mgr) {
                DEBUG_PRINT("freeing mem, pcount: %u, plimit: %u", mgr->pcount, mgr->plimit);
                --mgr->mcount;
            }
            free ( self );
        }
        else
        {
            DEBUG_PRINT("pooling mem, pcount: %u, plimit: %u", mgr->pcount, mgr->plimit);
            DLListPushTail ( & mgr -> mpool, & self -> n );
            ++ mgr -> pcount;
        }
    }
}

/*--------------------------------------------------------------------------
 * VDBManager
 *  opaque handle to library
 */

/* Whack
 */
static
rc_t VDBManagerWhack ( VDBManager *self )
{
    rc_t rc;

    KRefcountWhack ( & self -> refcount, "VDBManager" );

    rc = KDBManagerRelease ( self -> kmgr );
    if ( rc == 0 )
    {
        /* release user data */
        if ( self -> user != NULL && self -> user_whack != NULL )
        {
            ( * self -> user_whack ) ( self -> user );
            self -> user = NULL;
            self -> user_whack = NULL;
        }

        VSchemaRelease ( self -> schema );
        VLinkerRelease ( self -> linker );
        DLListWhack ( & self -> mpool, VDBMemWhack, NULL );
        free ( self );
        return 0;
    }

    KRefcountInit ( & self -> refcount, 1, "VDBManager", "whack", "vmgr" );
    return rc;
}


/* AddRef
 * Release
 *  all objects are reference counted
 *  NULL references are ignored
 */
LIB_EXPORT rc_t CC VDBManagerAddRef ( const VDBManager *self )
{
    if ( self != NULL )
    {
        switch ( KRefcountAdd ( & self -> refcount, "VDBManager" ) )
        {
        case krefLimit:
            return RC ( rcVDB, rcMgr, rcAttaching, rcRange, rcExcessive );
        }
    }
    return 0;
}

LIB_EXPORT rc_t CC VDBManagerRelease ( const VDBManager *self )
{
    if ( self != NULL )
    {
        switch ( KRefcountDrop ( & self -> refcount, "VDBManager" ) )
        {
        case krefWhack:
            return VDBManagerWhack ( ( VDBManager* ) self );
        case krefLimit:
            return RC ( rcVDB, rcMgr, rcReleasing, rcRange, rcExcessive );
        }
    }
    return 0;
}

/* Attach
 * Sever
 *  internal reference management
 */
VDBManager *VDBManagerAttach ( const VDBManager *self )
{
    if ( self != NULL )
    {
        switch ( KRefcountAddDep ( & self -> refcount, "VDBManager" ) )
        {
        case krefLimit:
            return NULL;
        }
    }
    return ( VDBManager* ) self;
}

rc_t VDBManagerSever ( const VDBManager *self )
{
    if ( self != NULL )
    {
        switch ( KRefcountDropDep ( & self -> refcount, "VDBManager" ) )
        {
        case krefWhack:
            return VDBManagerWhack ( ( VDBManager* ) self );
        case krefLimit:
            return RC ( rcVDB, rcMgr, rcReleasing, rcRange, rcExcessive );
        }
    }
    return 0;
}


/* Writable
 *  returns 0 if object is writable
 *  or a reason why if not
 *
 *  "path" [ IN ] - NUL terminated path
 */
LIB_EXPORT rc_t CC VDBManagerVWritable ( const VDBManager *self, const char *path, va_list args )
{
    if ( self == NULL )
        return RC ( rcVDB, rcMgr, rcAccessing, rcSelf, rcNull );
    return KDBManagerVWritable ( self -> kmgr, path, args );
}

LIB_EXPORT rc_t CC VDBManagerWritable ( const VDBManager *self, const char *path, ... )
{
    rc_t rc;

    va_list args;
    va_start ( args, path );

    rc = VDBManagerVWritable ( self, path, args );

    va_end ( args );

    return rc;
}


/* AddSchemaIncludePath
 *  add an include path to schema for locating input files
 */
LIB_EXPORT rc_t CC VDBManagerVAddSchemaIncludePath ( const VDBManager *self, const char *path, va_list args )
{
    if ( self != NULL )
        return VSchemaVAddIncludePath ( self -> schema, path, args );

    return RC ( rcVDB, rcMgr, rcUpdating, rcSelf, rcNull );
}

LIB_EXPORT rc_t CC VDBManagerAddSchemaIncludePath ( const VDBManager *self, const char *path, ... )
{
    rc_t rc;
    va_list args;

    va_start ( args, path );
    rc = VDBManagerVAddSchemaIncludePath ( self, path, args );
    va_end ( args );

    return rc;
}


/* AddLoadLibraryPath
 *  add a path[s] to loader for locating dynamic libraries
 */
LIB_EXPORT rc_t CC VDBManagerVAddLoadLibraryPath ( const VDBManager *self, const char *path, va_list args )
{
    if ( self != NULL )
        return VLinkerVAddLoadLibraryPath ( self -> linker, path, args );

    return RC ( rcVDB, rcMgr, rcUpdating, rcSelf, rcNull );
}

LIB_EXPORT rc_t CC VDBManagerAddLoadLibraryPath ( const VDBManager *self, const char *path, ... )
{
    rc_t rc;
    va_list args;

    va_start ( args, path );
    rc = VDBManagerVAddLoadLibraryPath ( self, path, args );
    va_end ( args );

    return rc;
}


/* ConfigFromKfg
 */
static
rc_t VDBManagerGetKfgPath ( const KConfig *kfg, const char *path, char *value, size_t value_size, size_t *num_read )
{
    /* open node */
    const KConfigNode *node;
    rc_t rc = KConfigOpenNodeRead ( kfg, & node, path );
    if ( rc == 0 )
    {
        size_t remaining;
        rc = KConfigNodeRead ( node, 0, value, value_size, num_read, & remaining );
        if ( rc == 0 )
        {
            if ( remaining != 0 )
                rc = RC ( rcVDB, rcMgr, rcConstructing, rcPath, rcExcessive );
            else if ( string_chr ( value, * num_read, '%' ) != NULL )
                rc = RC ( rcVDB, rcMgr, rcConstructing, rcPath, rcInvalid );
        }

        KConfigNodeRelease ( node );
    }
    return rc;
}

static
rc_t VDBManagerConfigFromKfg ( VDBManager *self, bool update )
{
    size_t num_read;
    char full [ 4096 ];

    /* open configuration manager */
    KConfig *kfg;
    rc_t rc = KConfigMake ( & kfg, NULL );
    if ( rc != 0 )
        rc = 0;
    else
    {
        /* look for load library paths */
        rc = VDBManagerGetKfgPath ( kfg, update ?
            "vdb/wmodule/paths": "vdb/module/paths",
            full, sizeof full, & num_read );
        if ( rc != 0 )
            rc = 0;
        else
        {
            /* split by ':' */
            const char * path = full;
            const char *colon = string_chr ( full, num_read, ':' );
            while ( colon != NULL )
            {
                /* add path between "path" and "colon" */
                rc = VDBManagerAddLoadLibraryPath ( self, "%.*s", ( int ) ( colon - path ), path );
                if ( rc != 0 )
                    break;
                num_read -= ( colon - path ) - 1;
                path = colon + 1;
                colon = string_chr ( path, num_read, ':' );
            }

            /* add in last portion */
            if ( rc == 0 && num_read != 0 )
                rc = VDBManagerAddLoadLibraryPath ( self, "%.*s", ( int ) ( num_read ), path );
        }

        /* look for schema paths */
        if ( rc == 0 )
            rc = VDBManagerGetKfgPath ( kfg, "vdb/schema/paths", full, sizeof full, & num_read );
        if ( rc != 0 )
            rc = 0;
        else
        {
            /* split by ':' */
            const char * path = full;
            const char *colon = string_chr ( full, num_read, ':' );
            while ( colon != NULL )
            {
                /* add path between "path" and "colon" */
                rc = VDBManagerAddSchemaIncludePath ( self, "%.*s", ( int ) ( colon - path ), path );
                if ( rc != 0 )
                    break;
                num_read -= ( colon - path ) - 1;
                path = colon + 1;
                colon = string_chr ( path, num_read, ':' );
            }

            /* add in last portion */
            if ( rc == 0 && num_read != 0 )
                rc = VDBManagerAddSchemaIncludePath ( self, "%.*s", ( int ) ( num_read ), path );
        }

        KConfigRelease ( kfg );
    }

    return rc;
}

/* ConfigFromLibPath
 *  trace this library back to file system
 *  attempt to locate relative paths
 *  add them to library and schema path lists
 */
static
rc_t VDBManagerConfigFromLibPath ( VDBManager *self, bool update )
{
    KDyld *dlmgr;
    rc_t rc = KDyldMake ( & dlmgr );
    if ( rc == 0 )
    {
        const KDirectory *home;
        rc = KDyldHomeDirectory ( dlmgr, & home, ( fptr_t ) VDBManagerConfigPaths );
        if ( rc == 0 )
        {
            char full [ 4096 ];

            /* paths to modules */
            const char *mod_paths [] =
            {
                "ncbi/mod",
                "vdb/mod",
                "../mod"
            };
            const char *wmod_paths [] =
            {
                "ncbi/wmod",
                "vdb/wmod",
                "../wmod"
            };

            /* paths to schema */
            const char *schema_paths [] =
            {
                "ncbi/schema",
                "vdb/schema",
                "../schema"
            };

            /* test for a few possible module paths */
            uint32_t i;
            const char **paths = update ? wmod_paths : mod_paths;
            for ( i = 0; rc == 0 && i < sizeof mod_paths / sizeof mod_paths [ 0 ]; ++ i )
            {
                uint32_t type = KDirectoryPathType ( home, "%s%u", paths [ i ], sizeof ( void* ) * 8 );
                if ( ( type & ~ kptAlias ) == kptDir )
                {
                    /* add full path */
                    rc = KDirectoryResolvePath ( home, true, full, sizeof full, "%s%u", paths [ i ], sizeof ( void* ) * 8 );
                    if ( rc == 0 )
                        rc = VDBManagerAddLoadLibraryPath ( self, full );
                }
            }

            for ( i = 0; rc == 0 && i < sizeof mod_paths / sizeof mod_paths [ 0 ]; ++ i )
            {
                uint32_t type = KDirectoryPathType ( home, paths [ i ] );
                if ( ( type & ~ kptAlias ) == kptDir )
                {
                    /* add full path */
                    rc = KDirectoryResolvePath ( home, true, full, sizeof full, paths [ i ] );
                    if ( rc == 0 )
                        rc = VDBManagerAddLoadLibraryPath ( self, full );
                }
            }

            for ( i = 0; rc == 0 && i < sizeof schema_paths / sizeof schema_paths [ 0 ]; ++ i )
            {
                uint32_t type = KDirectoryPathType ( home, schema_paths [ i ] );
                if ( ( type & ~ kptAlias ) == kptDir )
                {
                    /* add full path */
                    rc = KDirectoryResolvePath ( home, true, full, sizeof full, schema_paths [ i ] );
                    if ( rc == 0 )
                        rc = VDBManagerAddSchemaIncludePath ( self, full );
                }
            }

            KDirectoryRelease ( home );
        }

        KDyldRelease ( dlmgr );
    }
    return rc;
}

/* ConfigPaths
 *  looks for configuration information to set
 *  include paths for schema parser and
 *  load paths for linker
 */
rc_t VDBManagerConfigPaths ( VDBManager *self, bool update )
{
    rc_t rc = VDBManagerConfigFromKfg ( self, update );
    if ( rc == 0 )
        rc = VDBManagerConfigFromLibPath ( self, update );
    return rc;
}


/* GetUserData
 * SetUserData
 *  store/retrieve an opaque pointer to user data
 */
LIB_EXPORT rc_t CC VDBManagerGetUserData ( const VDBManager *self, void **data )
{
    rc_t rc;

    if ( data == NULL )
        rc = RC ( rcVDB, rcMgr, rcAccessing, rcParam, rcNull );
    else
    {
        if ( self == NULL )
            rc = RC ( rcVDB, rcMgr, rcAccessing, rcSelf, rcNull );
        else
        {
            * data = self -> user;
            return 0;
        }

        * data = NULL;
    }

    return rc;
}

LIB_EXPORT rc_t CC VDBManagerSetUserData ( const VDBManager *cself,
    void *data, void ( CC * destroy ) ( void *data ) )
{
    VDBManager *self = ( VDBManager* ) cself;
    if ( cself == NULL )
        return RC ( rcVDB, rcMgr, rcUpdating, rcSelf, rcNull );

    self -> user = data;
    self -> user_whack = destroy;

    return 0;
}


/* MakeMem
 *  pops a buffer from pool
 *  or allocates a new one on demand
 */
rc_t VDBManagerMakeMem ( VDBManager *self, VDBMem **memp )
{
    rc_t rc;

    if ( memp == NULL )
        rc = RC ( rcVDB, rcMgr, rcAllocating, rcParam, rcNull );
    else
    {
        if ( self == NULL )
            rc = RC ( rcVDB, rcMgr, rcAllocating, rcSelf, rcNull );
        else
        {
            VDBMem *mem;

            if ( self -> pcount > 0 )
            {
                /* "count" tells us that there is at least 1 pooled buffer */
                mem = ( VDBMem* ) DLListPopHead ( & self -> mpool );
                assert ( mem != NULL );

                /* should always happen */
                if ( mem != NULL )
                {
                    DEBUG_PRINT("retrieving mem from pool", 0);
                    -- self -> pcount;
                    * memp = mem;
                    return 0;
                }

                /* should never happen */
                self -> pcount = 0;
            }
            
            if (self->mcount > self->mlimit) {
                rc = RC(rcVDB, rcMgr, rcAllocating, rcResources, rcExhausted);
                DEBUG_PRINT("refusing to allocate; too many allocs", 0);
            }
            else {
                mem = malloc ( sizeof * mem );
                if ( mem == NULL )
                    rc = RC ( rcVDB, rcMgr, rcAllocating, rcMemory, rcExhausted );
                else
                {
                    ++self->mcount;
                    DEBUG_PRINT("allocating mem %u", self->mcount);
                    * memp = mem;
                    return 0;
                }
            }
        }

        * memp = NULL;
    }

    return rc;
}

/* OpenKDBManager
 *  returns a new reference to KDBManager used by VDBManager
 */
LIB_EXPORT rc_t CC VDBManagerOpenKDBManagerRead ( const VDBManager *self, const KDBManager **kmgr )
{
    rc_t rc;

    if ( kmgr == NULL )
        rc = RC ( rcVDB, rcMgr, rcAccessing, rcParam, rcNull );
    else
    {
        if ( self == NULL )
            rc = RC ( rcVDB, rcMgr, rcAccessing, rcSelf, rcNull );
        else
        {
            rc = KDBManagerAddRef ( self -> kmgr );
            if ( rc == 0 )
            {
                * kmgr = self -> kmgr;
                return 0;
            }
        }

        * kmgr = NULL;
    }

    return rc;
}
