/*==============================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include "align-info.vers.h"

#include <sra/sraschema.h> /* VDBManagerMakeSRASchema */

#include <vdb/manager.h> /* VDBManager */
#include <vdb/database.h> /* VDatabase */
#include <vdb/table.h> /* VTable */
#include <vdb/schema.h> /* VSchema */
#include <vdb/cursor.h> /* VCursor */

#include <kapp/main.h>

#include <kfg/config.h> /* KConfig */

#include <kdb/namelist.h> /* KMDataNodeListChild */
#include <kdb/meta.h> /* KMetadata */

#include <klib/container.h> /* BSTree */
#include <klib/sort.h> /* ksort */
#include <klib/out.h> /* OUTMSG */
#include <klib/log.h> /* LOGERR */
#include <klib/debug.h> /* DBGMSG */
#include <klib/rc.h> /* RC */

#include <assert.h>
#include <limits.h> /* PATH_MAX */
#include <stdio.h> /* sscanf */
#include <stdlib.h> /* free */
#include <string.h> /* strcmp */

#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

#define DISP_RC(rc, msg) (void)((rc == 0) ? 0 : LOGERR(klogInt, rc, msg))
#define DISP_RC2(rc, name, msg) (void)((rc == 0) ? 0 : \
    PLOGERR(klogInt, (klogInt,rc, "$(msg): $(name)","msg=%s,name=%s",msg,name)))
#define DESTRUCT(type, obj) do { rc_t rc2 = type##Release(obj); \
    if (rc2 && !rc) { rc = rc2; } obj = NULL; } while (false)

typedef struct Params {
    const char* dbPath;

    bool paramBamHeader;
    bool paramQuality;
    bool paramRef;
} Params;
typedef struct RefNode {
    BSTNode n;
    bool circular;
    char* name;
    char* seqId;
    bool row_lenCMP_READwas0;
} RefNode;
static rc_t ReadCfgStr
(const KConfig* kfg, const char* path, char* value, size_t value_sz)
{
    rc_t rc = 0;
    const KConfigNode *node = NULL;

    rc = KConfigOpenNodeRead(kfg, &node, path);
    if (rc == 0) {
        size_t num_read = 0;
        size_t remaining = 0;
        rc = KConfigNodeRead
            (node, 0, value, value_sz - 1, &num_read, &remaining);
        if (rc == 0) {
            if (remaining != 0)
            {   rc = RC(rcExe, rcConfig, rcReading, rcString, rcTooLong); }
            else { value[num_read] = '\0'; }
        }
    }

    DESTRUCT(KConfigNode, node);

    return rc;
}

typedef struct FindRefseq {
    const char* seqId;
    bool found;
} FindRefseq;
static rc_t CC scan_refseq_dir(const KDirectory* dir,
    uint32_t type, const char* name, void* data)
{
    FindRefseq* t = data;
    assert(dir && name && data && t->seqId);
    if (strcmp(t->seqId, name) == 0)
    {   t->found = true; }
    return 0;
}

static bool construct(rc_t* aRc, const KDirectory* native, const char* srv,
    const char* vol, const char* seqId, char* buf, size_t blen)
{
    rc_t rc = 0;

    const KDirectory* dir = NULL;

    FindRefseq t;

    assert(aRc && native && srv && seqId && buf && blen);

    rc = *aRc;

    if (rc != 0)
    {   return false; }

    if (vol)
    {   rc = KDirectoryOpenDirRead(native, &dir, false, "%s/%s", srv, vol); }
    else
    {   rc = KDirectoryOpenDirRead(native, &dir, false, srv); }

    if (rc == 0) {
        memset(&t, 0, sizeof t);
        t.seqId = seqId;
        rc = KDirectoryVVisit(dir, false, scan_refseq_dir, &t, ".", NULL);
    }

    if (rc == 0 && t.found) {
        if (vol) {
            rc = KDirectoryResolvePath
                (native, true, buf, blen, "%s/%s/%s", srv, vol, seqId);
        }
        else {
            rc = KDirectoryResolvePath
                (native, true, buf, blen, "%s/%s", srv, seqId);
        }
    }

    DESTRUCT(KDirectory, dir);

    *aRc = rc;

    return t.found;
}

static
rc_t privFindRef(const char* seqId, char* buf, size_t blen,
    bool destroy)
{
#define SIZE 4096
    rc_t rc = 0;
    static KDirectory* dir = NULL;
    static KConfig* cfg = NULL;
    static char Servers[SIZE] = "";
    static char Volumes[SIZE] = "";
    static char Paths[SIZE] = "";
    if (cfg == NULL && !destroy) {
        if (rc == 0) {
            rc = KDirectoryNativeDir(&dir);
            DISP_RC(rc, "while calling KDirectoryNativeDir");
        }
        if (rc == 0) {
            rc = KConfigMake(&cfg, NULL);
            DISP_RC(rc, "while calling KConfigMake");
        }
        if (rc == 0) {
            const char path[] = "refseq/servers";
            rc = ReadCfgStr(cfg, path, Servers, sizeof Servers);
            if (rc != 0) {
                if (GetRCState(rc) == rcNotFound)
                {   rc = 0; }
                else { DISP_RC2(rc, path, "while reading Node"); }
            }
            else {
                const char path[] = "refseq/volumes";
                rc = ReadCfgStr(cfg, path, Volumes, sizeof Volumes);
                if (rc != 0) {
                    if (GetRCState(rc) == rcNotFound)
                    {   rc = 0; }
                    else { DISP_RC2(rc, path, "while reading Node"); }
                }
            }
            if (rc == 0) {
                const char path[] = "refseq/paths";
                rc = ReadCfgStr(cfg, path, Paths, sizeof Paths);
                if (rc != 0) {
                    if (GetRCState(rc) == rcNotFound)
                    {   rc = 0; }
                    else { DISP_RC2(rc, path, "while reading Node"); }
                }
            }
        }
    }
    if (!destroy && rc == 0 && dir != NULL && cfg != NULL) {
        if (Servers[0] && Volumes[0])
        {
            char server[SIZE];
            char vol[SIZE];
            char* srv_sep = NULL;
            char *srv_rem = server;
            strcpy(server, Servers);
    /* TODO check for multiple servers/volumes */
            do {
                char *vol_rem = vol;
                char *vol_sep = NULL;
                strcpy(vol, Volumes);
                srv_sep = strchr(server, ':');
                if (srv_sep) {
                    srv_rem = srv_sep + 1;
                    *srv_sep = '\0';
                }
                do {
                    char const* volume = vol_rem;
                    vol_sep = strchr(volume, ':');
                    if (vol_sep) {
                        vol_rem = vol_sep + 1;
                        *vol_sep = 0;
                    }
                    if (construct(&rc, dir, server, volume,
                        seqId, buf, blen))
                    {   break; }
                } while(vol_sep && rc == 0);
            } while(srv_sep && rc == 0);
        }
        else if (Paths[0]) {
            char paths[SIZE];
            char *path_rem = paths;
            char *path_sep = NULL;
            strcpy(paths, Paths);
            do {
                char const* path = path_rem;
                path_sep = strchr(path, ':');
                if (path_sep) {
                    path_rem = path_sep + 1;
                    *path_sep = 0;
                }
                if (construct(&rc, dir, path, NULL, seqId, buf, blen))
                {   break; }
            } while(path_sep && rc == 0);
        }
    }
    if (destroy) {
        DESTRUCT(KConfig, cfg);
        DESTRUCT(KDirectory, dir);
    }
    return rc;
}

static rc_t DestroyConfig(void) { return privFindRef(NULL, NULL, 0, true); }

static rc_t FindRef(const char* seqId, char* buf, size_t blen)
{   return privFindRef(seqId, buf, blen, false); }

static int CC bstCmp(const void* item, const BSTNode* n) {
    const char* lhs = item;
    const RefNode* sn = (const RefNode*) n;

    assert(lhs && sn);

    return strcmp(lhs, sn->seqId);
}

static int CC bstSort(const BSTNode* item, const BSTNode* n) {
    const RefNode* sn = (const RefNode*) item;

    return bstCmp(sn->seqId, n);
}

static void CC bstPrint(BSTNode* n, void* data) {
    char buf[PATH_MAX + 1] = "";
    bool remote = false;
    const RefNode* sn = (const RefNode*) n;

    assert(sn);

    remote = sn->row_lenCMP_READwas0;
    if (remote) {
        rc_t rc = FindRef(sn->seqId, buf, sizeof buf);
        if (rc)
        {   buf[0] = '\0'; }
    }

    OUTMSG(("%s,%s,%s,%s", sn->seqId, sn->name,
        (sn->circular ? "true" : "false"), (remote ? "remote" : "local")));
    if (buf[0])
        {   OUTMSG((":%s", buf)); }
    OUTMSG(("\n"));
}

static void CC bstWhack(BSTNode* n, void* ignore) {
    RefNode* sn = (RefNode*) n;

    assert(sn);

    free(sn->name);
    sn->name = NULL;

    free(sn->seqId);
    sn->seqId = NULL;

    free(sn);
}

#define ALIAS_ALL    "a"
#define OPTION_ALL   "all"
static const char* USAGE_ALL[] = { "print all information", NULL };

#define ALIAS_BAM    "b"
#define OPTION_BAM   "bam"
static const char* USAGE_BAM[] = { "print bam header (if present)", NULL };

#define ALIAS_QUA    "q"
#define OPTION_QUA   "qual"
static const char* USAGE_QUA[]
                       = { "print quality statistics (if present)", NULL };

#define ALIAS_REF    "r"
#define OPTION_REF   "ref"
static const char* USAGE_REF[] = { "print refseq information [default]", NULL };

OptDef Options[] =
{
      { OPTION_ALL, ALIAS_ALL, NULL, USAGE_ALL, 1, false, false }
    , { OPTION_BAM, ALIAS_BAM, NULL, USAGE_BAM, 1, false, false }
    , { OPTION_QUA, ALIAS_QUA, NULL, USAGE_QUA, 1, false, false }
    , { OPTION_REF, ALIAS_REF, NULL, USAGE_REF, 1, false, false }
};

rc_t CC UsageSummary (const char * progname) {
    return KOutMsg (
        "Usage:\n"
        "  %s [options] <db-path>\n"
        "\n"
        "Summary:\n"
        "  Print database alignment information\n"
        "\n", progname);
 }

static const char* param_usage[] = { "Path to the database" };

rc_t CC Usage(const Args* args) { 
    rc_t rc = 0 ;

    const char* progname = UsageDefaultName;
    const char* fullpath = UsageDefaultName;

    if (args == NULL)
    {    rc = RC(rcExe, rcArgv, rcAccessing, rcSelf, rcNull); }
    else
    {    rc = ArgsProgram(args, &fullpath, &progname); }

    UsageSummary(progname);

    KOutMsg("Parameters:\n");

    HelpParamLine ("db-path", param_usage);

    KOutMsg ("\nOptions:\n");

    HelpOptionLine (ALIAS_ALL, OPTION_ALL, NULL, USAGE_ALL);
    HelpOptionLine (ALIAS_REF, OPTION_REF, NULL, USAGE_REF);
    HelpOptionLine (ALIAS_BAM, OPTION_BAM, NULL, USAGE_BAM);
    HelpOptionLine (ALIAS_QUA, OPTION_QUA, NULL, USAGE_QUA);

    HelpOptionsStandard ();

    HelpVersion (fullpath, KAppVersion());

    return rc;
}

const char UsageDefaultName[] = "align-info";

ver_t CC KAppVersion(void) { return ALIGN_INFO_VERS; }

static rc_t ArgsRelease(Args* self) { return ArgsWhack(self); }

typedef struct Column {
    uint32_t idx;
    const char* name;
} Column;

static rc_t AddColumn(rc_t rc, const VCursor* curs, Column* col) {
    if (rc == 0) {
        assert(col);

        rc = VCursorAddColumn(curs, &col->idx, col->name);
        DISP_RC2(rc, col->name, "while calling VCursorAddColumn");
    }

    return rc;
}

static rc_t CursorRead(rc_t rc, const VCursor* curs, int64_t row_id,
    const Column* col, char* buffer, uint32_t blen, uint32_t expected,
    uint32_t* row_len)
{
    if (rc == 0) {
        assert(col);

        if (blen == 0) {
            assert(buffer == NULL);
            ++blen;
        }

        rc = VCursorReadDirect
            (curs, row_id, col->idx, 8, buffer, blen - 1, row_len);

        if (rc != 0) {
            if (buffer == NULL &&
                rc == RC(rcVDB, rcCursor, rcReading, rcBuffer, rcInsufficient))
            {
                rc = 0;
            }
            else {
                PLOGERR(klogInt, (klogInt, rc,
                    "Error while calling VCursorReadDirect"
                    "($(name), row_len=$(row_len))",
                    "name=%s,row_len=%d", col->name, *row_len));
            }
        }
        else {
            if (expected > 0 && *row_len != expected) {
                rc = RC(rcExe, rcCursor, rcReading, rcData, rcUnexpected);
                PLOGERR(klogInt, (klogInt, rc,
                    "$(name) row_len = $(row_len)",
                    "name=%s,row_len=%d", col->name, row_len));
            }
            if (buffer != NULL)
            {   buffer[*row_len] = '\0'; }
        }
    }

    return rc;
}

static rc_t bam_header(const VDatabase* db) {
    rc_t rc = 0;
    const char path[] = "BAM_HEADER";
    const KMetadata* meta = NULL;
    const KMDataNode* node = NULL;
    char* buffer = NULL;
    assert(db);
    if (rc == 0) {
        rc = VDatabaseOpenMetadataRead(db, &meta);
        DISP_RC(rc, "while calling VDatabaseOpenMetadataRead");
    }
    if (rc == 0) {
        rc = KMetadataOpenNodeRead(meta, &node, path);
        if (GetRCState(rc) == rcNotFound)
        {   rc = 0; }
        else {
            DISP_RC2(rc, path, "while calling KMetadataOpenNodeRead");
            if (rc == 0) {
                int i = 0;
                size_t bsize = 0;
                size_t size = 1024;
                for (i = 0; i < 2; ++i) {
                    free(buffer);
                    bsize = size + 1;
                    buffer = malloc(bsize);
                    if (buffer == NULL) {
                        rc = RC(rcExe, rcStorage, rcAllocating,
                            rcMemory, rcExhausted);
                    }
                    else {
                        rc = KMDataNodeReadCString(node, buffer, bsize, &size);
                        if (rc == 0)
                        {   break; }
                        else if (i == 0 && GetRCObject(rc) == rcBuffer
                                        && GetRCState(rc) == rcInsufficient)
                        {   rc = 0; }
                    }
                    DISP_RC2(rc, path, "while calling KMDataNodeReadCString");
                }
            }
        }
    }
    if (rc == 0 && buffer)
    {   OUTMSG(("BAM_HEADER: {\n%s}\n\n", buffer)); }
    DESTRUCT(KMDataNode, node);
    DESTRUCT(KMetadata, meta);
    free(buffer);
    return rc;
}

static int CC sort_callback(const void* p1, const void* p2, void* data) {
    int i1 = *(int*) p1;
    int i2 = *(int*) p2;
    return i1 - i2;
}
static rc_t qual_stats(const Params* prm, const VDatabase* db) {
    rc_t rc = 0;
    const char tblName[] = "SEQUENCE";
    const VTable* tbl = NULL;
    const KMetadata* meta = NULL;
    const KMDataNode* node = NULL;
    assert(prm && db);
    if (rc == 0) {
        rc = VDatabaseOpenTableRead(db, &tbl, tblName);
        DISP_RC2(rc, tblName, "while calling VDatabaseOpenTableRead");
    }
    if (rc == 0) {
        rc = VTableOpenMetadataRead(tbl, &meta);
        DISP_RC2(rc, tblName, "while calling VTableOpenMetadataRead");
    }
    if (rc == 0) {
        bool found = false;
        const char path[] = "STATS/QUALITY";
        rc = KMetadataOpenNodeRead(meta, &node, path);
        if (rc == 0)
        {   found = true; }
        else if (GetRCState(rc) == rcNotFound)
        {   rc = 0; }
        DISP_RC2(rc, path, "while calling KMetadataOpenNodeRead");
        if (found) {
            uint32_t i = 0;
            int nbr = 0;
            uint32_t count = 0;
            KNamelist* names = NULL;
            int* quals = NULL;
            if (rc == 0) {
                rc = KMDataNodeListChild(node, &names);
                DISP_RC2(rc, path, "while calling KMDataNodeListChild");
            }
            if (rc == 0) {
                rc = KNamelistCount(names, &count);
                DISP_RC2(rc, path, "while calling KNamelistCount");
                if (rc == 0 && count > 0) {
                    quals = calloc(count, sizeof *quals);
                    if (quals == NULL) {
                        rc = RC(rcExe,
                            rcStorage, rcAllocating, rcMemory, rcExhausted);
                    }
                }
            }
            for (i = 0; i < count && rc == 0; ++i) {
             /* uint64_t u = 0;
                const KMDataNode* n = NULL; */
                const char* nodeName = NULL;
                const char* name = NULL;
                rc = KNamelistGet(names, i, &nodeName);
                DISP_RC2(rc, path, "while calling KNamelistGet");
                if (rc)
                {   break; }
                name = nodeName;
             /* rc = KMDataNodeOpenNodeRead(node, &n, name);
                DISP_RC(rc, name);
                if (rc == 0) {
                    rc = KMDataNodeReadAsU64(n, &u);
                    DISP_RC(rc, name);
                } */
                if (rc == 0) {
                    char* c = strchr(name, '_');
                    if (c != NULL && *(c + 1) != '\0') {
                        name = c + 1;
                        if (sscanf(name, "%d", &quals[i]) != 1) {
                            rc = RC(rcExe,
                                rcNode, rcParsing, rcName, rcUnexpected);
                            PLOGERR(klogInt,
                                (klogInt, rc, "$(name)", "name=%s", nodeName));
                        }
                    }
                    /* OUTMSG(("QUALITY %s %lu\n", name, u)); */
                }
             /* DESTRUCT(KMDataNode, n); */
            }
            if (rc == 0 && count > 0)
            {   ksort(quals, count, sizeof *quals, sort_callback, NULL); }
            if (rc == 0) {
                OUTMSG(("%s", prm->dbPath));
            }
            for (i = 0, nbr = 0; i < count && rc == 0; ++i, ++nbr) {
                uint64_t u = 0;
                char name[64];
                const KMDataNode* n = NULL;
                sprintf(name, "PHRED_%d", quals[i]);
                rc = KMDataNodeOpenNodeRead(node, &n, name);
                DISP_RC(rc, name);
                if (rc == 0) {
                    rc = KMDataNodeReadAsU64(n, &u);
                    DISP_RC(rc, name);
                    if (rc == 0) {
                        while (nbr < quals[i]) {
                            OUTMSG(("\t0"));
                            ++nbr;
                        }
                        OUTMSG(("\t%lu", u));
                    /*  OUTMSG(("QUALITY %d %lu\n", quals[i], u)); */
                    }
                }
                DESTRUCT(KMDataNode, n);
            }
            while (rc == 0 && nbr <= 40) {
                OUTMSG(("\t0"));
                nbr++;
            }
            if (rc == 0) {
                OUTMSG(("\n"));
            }
            DESTRUCT(KNamelist, names);
        }
    }
    DESTRUCT(KMDataNode, node);
    DESTRUCT(KMetadata, meta);
    DESTRUCT(VTable, tbl);
    return rc;
}

static rc_t Valign_info(const VDatabase* db, BSTree* tr) {
    rc_t rc = 0;
    int64_t i = 0;
    int64_t firstId = 0;
    int64_t lastId = 0;
    const char tblName[] = "REFERENCE";
    Column CIRCULAR = { 0, "CIRCULAR" };
    Column CMP_READ = { 0, "CMP_READ" };
    Column NAME     = { 0, "NAME"     };
    Column SEQ_ID   = { 0, "SEQ_ID"   };
    const VCursor* curs = NULL;
    const VTable* tbl = NULL;
    if (db == NULL || tr == NULL)
    {   return RC(rcExe, rcQuery, rcExecuting, rcParam, rcNull); }
    if (rc == 0) {
        rc = VDatabaseOpenTableRead(db, &tbl, tblName);
        DISP_RC2(rc, tblName, "while calling VDatabaseOpenTableRead");
    }
    if (rc == 0) {
        rc = VTableCreateCursorRead(tbl, &curs);
        DISP_RC2(rc, tblName, "while calling VTableCreateCursorRead");
    }
    rc = AddColumn(rc, curs, &CIRCULAR);
    rc = AddColumn(rc, curs, &CMP_READ);
    rc = AddColumn(rc, curs, &NAME);
    rc = AddColumn(rc, curs, &SEQ_ID);
    if (rc == 0) {
        rc = VCursorOpen(curs);
        DISP_RC2(rc, tblName, "while calling VCursorOpen");
    }
    if (rc == 0) {
        uint64_t count = 0;
        rc = VCursorIdRange(curs, 0, &firstId, &count);
        DISP_RC2(rc, tblName, "while calling VCursorIdRange");
        lastId = firstId + count - 1;
    }
    for (i = firstId; i <= lastId && rc == 0; ++i) {
        uint32_t row_len = 0;
        uint32_t row_lenCMP_READ = 0;
        char circular[8] = "";
        char name[256] = "";
        char seqId[256] = "";
        rc = CursorRead(rc, curs, i, &CMP_READ, NULL, 0, 0, &row_lenCMP_READ);
        rc = CursorRead(rc, curs, i, &CIRCULAR, circular, sizeof circular, 1,
            &row_len);
        rc = CursorRead(rc, curs, i, &SEQ_ID, seqId, sizeof seqId, 0, &row_len);
        rc = CursorRead(rc, curs, i, &NAME, name, sizeof name, 0, &row_len);
        if (rc == 0) {
            RefNode* sn = NULL;
            DBGMSG(DBG_APP, DBG_COND_1, ("SEQ_ID: %s\n", seqId));
            sn = (RefNode*) BSTreeFind(tr, seqId, bstCmp);
            if (sn == NULL) {
                sn = calloc(1, sizeof(*sn));
                if (sn == NULL) {
                    rc = RC
                        (rcExe, rcStorage, rcAllocating, rcMemory, rcExhausted);
                    break;
                }
                sn->seqId = strdup(seqId);
                if (sn->seqId == NULL) {
                    bstWhack((BSTNode*) sn, NULL);
                    sn = NULL;
                    rc = RC
                        (rcExe, rcStorage, rcAllocating, rcMemory, rcExhausted);
                    break;
                }
                sn->name = strdup(name);
                if (sn->name == NULL) {
                    bstWhack((BSTNode*) sn, NULL);
                    sn = NULL;
                    rc = RC
                        (rcExe, rcStorage, rcAllocating, rcMemory, rcExhausted);
                    break;
                }
                sn->circular = circular[0];
                sn->row_lenCMP_READwas0 = (row_lenCMP_READ == 0);
                BSTreeInsert(tr, (BSTNode*)sn, bstSort);
            }
            else {
                if (row_lenCMP_READ == 0)
                {   sn->row_lenCMP_READwas0 = true; }
                if (strcmp(sn->name, name) || sn->circular != circular[0]) {
                    rc = RC(rcExe, rcCursor, rcReading, rcData, rcInconsistent);
                    PLOGERR(klogInt, (klogInt, rc,
                        "while reading row $(row_id)", "row_id=%d", i));
                }
            }
        }
    }
    DESTRUCT(VTable, tbl);
    DESTRUCT(VCursor, curs);
    return rc;
}

static rc_t align_info(const Params* prm) {
    rc_t rc = 0;
    const VDatabase* db = NULL;
    const VDBManager* mgr = NULL;
    VSchema* schema = NULL;
    if (prm == NULL)
    {   return RC(rcExe, rcQuery, rcExecuting, rcParam, rcNull); }
    if (rc == 0) {
        rc = VDBManagerMakeRead(&mgr, NULL);
        DISP_RC(rc, "while calling VDBManagerMakeRead");
    }
    if (rc == 0) {
        rc = VDBManagerMakeSRASchema(mgr, &schema);
        DISP_RC(rc, "while calling VDBManagerMakeSRASchema");
    }
    if (rc == 0) {
/*      const char path[] = "/panfs/pan1/trace_work/scratch/XXX000013"; */
        rc = VDBManagerOpenDBRead(mgr, &db, schema, prm->dbPath);
        if (rc)
	{   PLOGERR(klogErr,(klogErr, rc, "$(path)", "path=%s", prm->dbPath)); }
    }
    if (rc == 0) {
        if (prm->paramRef) {
            BSTree tr;
            BSTreeInit(&tr);
            rc = Valign_info(db, &tr);
            if (rc == 0) {
                BSTreeForEach(&tr, false, bstPrint, NULL);
                OUTMSG(("\n"));
            }
            BSTreeWhack(&tr, bstWhack, NULL);
            {
                rc_t rc2 = DestroyConfig();
                if (rc2 != 0 && rc == 0)
                {   rc = rc2; }
            }
        }
        if (prm->paramBamHeader) {
            rc_t rc3 = bam_header(db);
            if (rc3 != 0 && rc == 0)
            {   rc = rc3; }
        }
        if (prm->paramQuality) {
            rc_t rc3 = qual_stats(prm, db);
            if (rc3 != 0 && rc == 0)
            {   rc = rc3; }
        }
    }
    DESTRUCT(VSchema, schema);
    DESTRUCT(VDBManager, mgr);
    DESTRUCT(VDatabase, db);
    return rc;
}

rc_t CC KMain(int argc, char* argv[]) {
    rc_t rc = 0;
    Args* args = NULL;

    Params prm;
    memset(&prm, 0, sizeof prm);

    do {
        uint32_t pcount = 0;

        rc = ArgsMakeAndHandle(&args, argc, argv, 1,
            Options, sizeof Options / sizeof (OptDef));
        if (rc) {
            LOGERR(klogErr, rc, "While calling ArgsMakeAndHandle");
            break;
        }
        rc = ArgsParamCount(args, &pcount);
        if (rc) {
            LOGERR(klogErr, rc, "While calling ArgsParamCount");
            break;
        }
        if (pcount < 1) {
            rc = RC(rcExe, rcArgv, rcParsing, rcParam, rcInsufficient);
            MiniUsage(args);
            break;
        }
        if (pcount > 1) {
            rc = RC(rcExe, rcArgv, rcParsing, rcParam, rcAmbiguous);
            LOGERR(klogErr, rc, "Too many database parameters");
            break;
        }
        rc = ArgsParamValue(args, 0, &prm.dbPath);
        if (rc) {
            LOGERR(klogErr, rc, "Failure retrieving database name");
            break;
        }

        rc = ArgsOptionCount (args, OPTION_ALL, &pcount);
        if (rc) {
            LOGERR(klogErr, rc, "Failure to get '" OPTION_ALL "' argument");
            break;
        }
        if (pcount)
        {   prm.paramBamHeader = prm.paramQuality = prm.paramRef = true; }

        rc = ArgsOptionCount (args, OPTION_BAM, &pcount);
        if (rc) {
            LOGERR(klogErr, rc, "Failure to get '" OPTION_BAM "' argument");
            break;
        }
        if (pcount)
        {   prm.paramBamHeader = true; }

        rc = ArgsOptionCount (args, OPTION_QUA, &pcount);
        if (rc) {
            LOGERR(klogErr, rc, "Failure to get '" OPTION_QUA "' argument");
            break;
        }
        if (pcount)
        {   prm.paramQuality = true; }

        rc = ArgsOptionCount (args, OPTION_REF, &pcount);
        if (rc) {
            LOGERR(klogErr, rc, "Failure to get '" OPTION_REF "' argument");
            break;
        }
        if (pcount)
        {   prm.paramRef = true; }

        if (!prm.paramBamHeader && !prm.paramQuality && !prm.paramRef)
        {   prm.paramRef = true; }
    } while (false);

    if (rc == 0)
    {   rc = align_info(&prm); }

    DESTRUCT(Args, args);
    return rc;
}

/************************************* EOF ************************************/
